//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.4

import wesual.Controls      1.0
import wesual.data.Core     1.0

Item {
    id : deleteHandler


    readonly property Context context : documents.length > 0 ? documents[0].context : null
    property list<Document> documents

    property string displayProperty : "label"

    property string singleTitle : qsTrId("5da5bc715819eb08")
    property string singleInstructionText : qsTrId("690fb7380e2ff280")

    property string multipleTitle : qsTrId("3353abf235216f8b")
    property string multipleInstructionText : qsTrId("37079b78340e509b")

    signal accepted(var deletedDocuments)
    signal rejected()

    QtObject {
        id : p_

        property list<ReferenceCount> refCounts

        /*! \brief counts all documents that are referenced */
        property int conflictCount : 0

        /*!
            \brief stores the single document
        */
        property Document singleDocument : null

        property var replaceMap : {
            "alldocs"        : documents.length,
            "conflictcount"  : conflictCount,
            "label"          : firstLabelElided.elidedText,
            "type"           : documents.length > 0 ? documents[0].typeName : ""
        }

        function deleteUnused() {
            var deleted = [];
            documents[0].beginRecordChanges();
            for (var i = 0; i < refCounts.length; ++i) {
                if (refCounts[i].count === 0) {
                    deleted.push(refCounts[i].document);
                    refCounts[i].document.deleted = true;
                }
            }
            documents[0].endRecordChanges();
            deleteHandler.accepted(deleted);
            cleanup();
        }

        function deleteAll() {
            if (documents.length < 0)
                return;
            var deleted = [];
            documents[0].beginRecordChanges();
            for (var i = 0; i < refCounts.length; ++i) {
                deleted.push(refCounts[i].document);
                refCounts[i].document.deleted = true;
            }
            documents[0].endRecordChanges();
            deleteHandler.accepted(deleted);
            cleanup();
        }

        function cleanup() {
            deleteHandler.destroy();
        }
    }

    TextMetrics {
        id : firstLabelElided
        elideWidth: 200
        elide: Text.ElideMiddle
        text : {
            if (documents.length === 0)
                return ""

            for (var i = 0; i < p_.refCounts.length; ++i) {
                var refObj = p_.refCounts[i];
                if (refObj.count > 0) {
                    return refObj.document[displayProperty]
                }
            }
            return "";
        }

    }



    function initialize() {
        var refObjects = [];
        for (var i = 0; i < documents.length; ++i) {
            var doc = documents[i];
            var ref = refComponent.createObject(deleteHandler);
            ref.document = doc;
            refObjects.push(ref);
        }
        p_.refCounts = refObjects;
    }

    function refCountFinished() {
        var count = 0;
        var referenceCount = 0;
        var singleDoc = null;
        for (var i = 0; i < p_.refCounts.length; ++i) {
            var refObj = p_.refCounts[i];
            if (refObj.status !== ReferenceCount.Ready) {
                return false;
            }

            if (refObj.count > 0) {
                referenceCount = refObj.count;
                singleDoc = refObj.document;
                count++;
            }
        }
        p_.conflictCount = count

        singleMessageBox.title = cleanupText(singleTitle);
        singleMessageBox.instructionText = cleanupText(singleInstructionText);

        if (p_.conflictCount == 1) {
            multipleMessageBox.title = cleanupText(singleTitle);
        } else {
            multipleMessageBox.title = cleanupText(multipleTitle);
        }

        multipleMessageBox.instructionText = cleanupText(multipleInstructionText);


        return true;
    }

    function cleanupText(text) {
        return text.replace(/{([A-Za-z]+)}/gi, function(fullmatch, match) {
            if (p_.replaceMap[match.toLowerCase()]) {
                return p_.replaceMap[match.toLowerCase()];
            }
            return "";
        });
    }

    Component.onCompleted: {
        initialize();
    }
    Connections {
        target: context
        onBusyChanged : {
            if (!context.busy) {
                var finished = refCountFinished();
                if (finished) {
                    // only on document conflicted
                    if (documents.length === 1 && p_.conflictCount === 1) {
                        singleMessageBox.show();
                    } else if (p_.conflictCount >= 1) {
                        multipleMessageBox.show();
                    } else {
                        p_.deleteAll();
                    }
                }

            }
        }
    }

    Component {
        id : refComponent
        ReferenceCount {
        }
    }

    MessageBox {
        id : singleMessageBox

        primaryButtonIndex : 1
        type : DialogItem.TypeWarning

        contentWidth : 550

        buttons : customButtons
        property list<DialogButton> customButtons : [
            DialogButton {
                text : qsTrId("067ce595b65acd59")
                onClicked : {
                    singleMessageBox.accept(true);
                    p_.deleteAll();
                }
            },
            DialogButton {
                text : qsTrId("37ae1dd525a0ce7a")
                onClicked : {
                    singleMessageBox.reject();
                }
            }
        ]

        onRejected : {
            deleteHandler.rejected()
            p_.cleanup();
        }
    }

    MessageBox {
        id : multipleMessageBox

        contentWidth : 550

        primaryButtonIndex : 1
        type : DialogItem.TypeWarning

        buttons : customButtons
        property list<DialogButton> customButtons : [
            DialogButton {
                text : qsTrId("9ce7df2b56b023ee")
                onClicked : {
                    multipleMessageBox.accept(true);
                    p_.deleteAll();
                }
            },
            DialogButton {
                text : qsTrId("fa8b79d872bafa06")
                onClicked : {
                    multipleMessageBox.accept(true);
                    p_.deleteUnused();
                }
            },
            DialogButton {
                text : qsTrId("37ae1dd525a0ce7a")
                onClicked : {
                    multipleMessageBox.reject();
                }
            }
        ]

        onRejected : {
            deleteHandler.rejected()
            p_.cleanup();
        }
    }
}

